from sqlalchemy.orm import relationship

from sqlalchemy import Column, ForeignKey, Float, Integer, String, Text

from utils.postgres_database import Base


## connecting table for M:N relation
class LectureTopic(Base):
    __tablename__ = 'lecture_topics'

    id = Column(Integer, primary_key=True, index=True, autoincrement=True)
    lecture_id = Column(Integer, ForeignKey('lectures.id'))
    topic_id = Column(Integer, ForeignKey('topics.id'))
    topic_probability = Column(Float, default=0.0)

    lecture = relationship("Lecture", back_populates="topics")
    topic = relationship("Topic", back_populates="lectures")

# Model for table lectures
class Lecture(Base):
    __tablename__ = 'lectures'

    id = Column(Integer, primary_key=True, index=True, autoincrement=True) # indexable, it means increase the performance
    text = Column(Text, nullable=False)
    summary = Column(Text, nullable=False)
    duration_seconds = Column(Integer, nullable=False)
    lecturer = Column(String(20), nullable=True) # F or M or null, we can have video without a sound
    voice_language = Column(String(20), nullable=True) # eng
    slides_language = Column(String(20), nullable=False)

    # 1:N relation on table "images"
    images = relationship('Image', back_populates='lecture')

    # 1:N relation on table "vdu_data"
    visual_doc_understanding_data = relationship('VisualDocUnderstandingData', back_populates='lecture')

    # 1:N relation on association table "lecture_topics"
    topics = relationship('LectureTopic', back_populates='lecture')


# Model for table imagges
class Image(Base):
    __tablename__ = 'images'

    id = Column(Integer, primary_key=True, index=True, autoincrement=True)
    image_url = Column(String(512), nullable=False)
    image_type = Column(String(256))
    lecture_id = Column(Integer, ForeignKey("lectures.id"))
    lecture = relationship('Lecture', back_populates='images')

# Model for table visual_doc_understanding_data
class VisualDocUnderstandingData(Base):
    __tablename__ = "visual_doc_understanding_data"

    id = Column(Integer, primary_key=True, index=True, autoincrement=True)
    vdu_output = Column(Text, nullable=True)
    vdu_model = Column(String(128), nullable=True)
    lecture_id = Column(Integer, ForeignKey("lectures.id"))
    lecture = relationship('Lecture', back_populates='visual_doc_understanding_data')


# Model for table topics
class Topic(Base):
    __tablename__ = 'topics'

    id = Column(Integer, primary_key=True, index=True, autoincrement=True)
    topic_category = Column(String(256), nullable=False, unique=True)
    lectures = relationship('LectureTopic', back_populates='topic')





